module type S = 
sig
  type t
  val copy: t -> t
  val equal: t -> t -> bool
  val hash: t -> int
  val compare: t -> t -> int
end

module Std =
struct 
(*  let copy x = Obj.obj (Obj.dup (Obj.repr x)) *)
  let copy x = x
  let equal x y = (compare x y) = 0
  let hash = Hashtbl.hash
  let compare = Pervasives.compare
end

module Lift (Base: S)  =
struct
  type t = Bot | Lifted of Base.t | Top
  include Std

  let copy x =
    match x with
      | Lifted n -> Lifted (Base.copy n)
      | tb -> tb

  let equal x y = 
    match (x, y) with
      | (Top, Top) -> true
      | (Bot, Bot) -> true
      | (Lifted x, Lifted y) -> Base.equal x y
      | _ -> false

end

module Lift2 (Base1: S) (Base2: S)  =
struct
  type t = Bot | Lifted1 of Base1.t | Lifted2 of Base2.t | Top
  include Std

  let copy x =
    match x with
      | Lifted1 n -> Lifted1 (Base1.copy n)
      | Lifted2 n -> Lifted2 (Base2.copy n)
      | tb -> tb

  let equal x y = 
    match (x, y) with
      | (Top, Top) -> true
      | (Bot, Bot) -> true
      | (Lifted1 x, Lifted1 y) -> Base1.equal x y
      | (Lifted2 x, Lifted2 y) -> Base2.equal x y
      | _ -> false

end


module Prod (Base1: S) (Base2: S) =
struct
  type t = Base1.t * Base2.t
  include Std
  let copy (x,y) = (Base1.copy x, Base2.copy y)
  let equal (x1,x2) (y1,y2) = Base1.equal x1 y1 && Base2.equal x2 y2
end
  
module Prod3 (Base1: S) (Base2: S) (Base3: S) =
struct
  type t = Base1.t * Base2.t * Base3.t
  include Std
  let copy (x,y,z) = (Base1.copy x, Base2.copy y, Base3.copy z)
  let equal (x1,x2,x3) (y1,y2,y3) = 
    Base1.equal x1 y1 && Base2.equal x2 y2 && Base3.equal x3 y3
end
  

module InfMap (Domain: S) (Range: S) =
struct
  module M = Map.Make(Domain)
  include Std

  type key = Domain.t
  type value = Range.t
  type t = Range.t M.t * Range.t 
             (* (map: Domain->Range, default: Range) *)
  let is_simple _ = false

  let mem (mapping, _ ) key = M.mem key mapping

  let find (mapping, defval) key = try M.find key mapping with Not_found -> defval

  (*
  let replace (mapping, defval) key value = 
    if Range.equal value defval then
      M.remove mapping key
    else 
      M.replace mapping key value

  let add (mapping, defval : t) key value = 
    if not (Range.equal value defval) then M.add mapping key value
  *)
  let addforced (mapping, defval) key value = 
    (M.add key value mapping, defval)
      
  let add = addforced
(*
  let replace (mapping, defval) key value = 
    M.replace mapping key value
*)
  let replace = add

(*
  let copy_map mp =
    let res = M.create 113  in
    let add_fun key value = M.add res key (Range.copy value) in
      M.iter add_fun mp;
      res
  let copy (mp, df)  = (copy_map mp, Range.copy df)
*)
  let copy_map x = x

  exception Result_known
  let equal (xm,xd as x:t) (ym,yd as y:t) = 
    if Range.equal xd yd then
      let test other key value = 
        if Range.equal value (find other key) then () else raise Result_known
      in try 
        M.iter (test x) ym;
        M.iter (test y) xm;
        true;
      with Result_known -> false
    else 
      false
end

module InfSet (Base: S) = 
struct
  module S = Set.Make(Base)
  type t = All | Set of S.t
  type elt = Base.t
  include Std
  let copy x = x

  let empty = Set S.empty
  let is_empty x = 
    match x with
      | All -> false 
      | Set x -> S.is_empty x
  let mem x s = 
    match s with
      | All -> true
      | Set s -> S.mem x s
  let add x s = 
    match s with
      | All -> All
      | Set s -> Set (S.add x s)
  let singleton x = Set (S.singleton x)
  let remove x s = 
    match s with 
      | All -> All   (* NB! NB! NB! *)
      | Set s -> Set (S.remove x s)
  let union x y = 
    match x, y with
      | All, _ -> All
      | _, All -> All
      | Set x, Set y -> Set (S.union x y)
  let inter x y = 
    match x, y with
      | All, y -> y
      | x, All -> x
      | Set x, Set y -> Set (S.inter x y)
  let diff x y = 
    match x, y with
      | x, All -> empty
      | All, y -> All (* NB! NB! NB! *)
      | Set x, Set y -> Set (S.diff x y)
  let subset x y =
    match x, y with
      | _, All -> true
      | All, _ -> false
      | Set x, Set y -> S.subset x y  
  let iter f x =
    match x with
      | All -> ()
      | Set x -> S.iter f x
  let fold f allval x unitval = 
    match x with
      | All -> allval
      | Set x -> S.fold f x unitval
  let map f x = 
    match x with 
      | All -> All
      | Set s -> let g x rest = S.add (f x) rest in Set (S.fold g s S.empty)
  let partition f x = 
    match x with
      | All -> (All,All) (* NB! NB! NB! *)
      | Set x -> let (x1,x2) = (S.partition f x) in (Set x1, Set x2)
  exception Choose_all
  let choose x = 
    match x with
      | All -> raise Choose_all
      | Set x -> S.choose x

end

module Sequence (Base: S): S with type t = Base.t list = 
struct
  type t = Base.t list
  include Std
  let copy = List.map Base.copy
  let equal x y = try List.for_all2 Base.equal x y with Invalid_argument _ -> false
end
