module type S =
sig
  include Printable.S
  val leq: t -> t -> bool
  val join: t -> t -> t
  val meet: t -> t -> t
  val bot: unit -> t
  val top: unit -> t
end

exception Unsupported
module Fake (Base: Printable.S) = 
struct 
  include Base
  let leq = equal
  let join x y = if equal x y then x else raise Unsupported
  let meet x y = if equal x y then x else raise Unsupported
  let bot () = raise Unsupported
  let top () = raise Unsupported
end

module Reverse (Base: S) =
struct
  include Base
  let bot = Base.top
  let top = Base.bot
  let leq x y = Base.leq y x
  let join = Base.meet
  let meet = Base.join
end

module Flat (Base: Printable.S) = 
struct 
  include Printable.Lift(Base)
  let _ = liftname := "order flattened"

  let bot () = Bot
  let top () = Top

  let leq x y =
    match (x,y) with
      | (_, Top) -> true
      | (Top, _) -> false
      | (Bot, _) -> true
      | (_, Bot) -> false
      | (Lifted x, Lifted y) -> Base.equal x y

  let join x y = 
    match (x,y) with 
      | (Top, _) -> Top
      | (_, Top) -> Top
      | (Bot, x) -> x
      | (x, Bot) -> x
      | (Lifted x, Lifted y) when Base.equal x y -> Lifted x
      | _ -> Top

  let meet x y = 
    match (x,y) with 
      | (Bot, _) -> Bot
      | (_, Bot) -> Bot
      | (Top, x) -> x
      | (x, Top) -> x
      | (Lifted x, Lifted y) when Base.equal x y -> Lifted x
      | _ -> Bot
end


module Lift (Base: S) = 
struct 
  include Printable.Lift(Base) 

  let bot () = Bot
  let top () = Top

  let leq x y =
    match (x,y) with
      | (_, Top) -> true
      | (Top, _) -> false
      | (Bot, _) -> true
      | (_, Bot) -> false
      | (Lifted x, Lifted y) -> Base.leq x y

  let join x y = 
    match (x,y) with 
      | (Top, _) -> Top
      | (_, Top) -> Top
      | (Bot, x) -> x
      | (x, Bot) -> x
      | (Lifted x, Lifted y) -> Lifted (Base.join x y)

  let meet x y = 
    match (x,y) with 
      | (Bot, _) -> Bot
      | (_, Bot) -> Bot
      | (Top, x) -> x
      | (x, Top) -> x
      | (Lifted x, Lifted y) -> Lifted (Base.meet x y)
end

module Lift2 (Base1: S) (Base2: S) = 
struct 
  include Printable.Lift2 (Base1) (Base2)

  let bot () = Bot
  let top () = Top

  let leq x y =
    match (x,y) with
      | (_, Top) -> true
      | (Top, _) -> false
      | (Bot, _) -> true
      | (_, Bot) -> false
      | (Lifted1 x, Lifted1 y) -> Base1.leq x y
      | (Lifted2 x, Lifted2 y) -> Base2.leq x y
      | _ -> false

  let join x y = 
    try match (x,y) with 
      | (Top, _) -> Top
      | (_, Top) -> Top
      | (Bot, x) -> x
      | (x, Bot) -> x
      | (Lifted1 x, Lifted1 y) -> Lifted1 (Base1.join x y)
      | (Lifted2 x, Lifted2 y) -> Lifted2 (Base2.join x y)
      | _ -> Top
    with
      | Unsupported -> Top

  let meet x y = 
    try match (x,y) with 
      | (Bot, _) -> Bot
      | (_, Bot) -> Bot
      | (Top, x) -> x
      | (x, Top) -> x
      | (Lifted1 x, Lifted1 y) -> Lifted1 (Base1.meet x y)
      | (Lifted2 x, Lifted2 y) -> Lifted2 (Base2.meet x y)
      | _ -> Bot
    with
      | Unsupported -> Bot
end

module Prod (Base1: S) (Base2: S) =
struct
  include Printable.Prod (Base1) (Base2)

  let bot () = (Base1.bot (), Base2.bot ())
  let top () = (Base1.top (), Base2.top ())

  let leq (x1,x2) (y1,y2) = Base1.leq x1 y1 && Base2.leq x2 y2

  let op_scheme op1 op2 (x1,x2) (y1,y2): t = (op1 x1 y1, op2 x2 y2)
  let join = op_scheme Base1.join Base2.join
  let meet = op_scheme Base1.meet Base2.meet
end

module Prod3 (Base1: S) (Base2: S) (Base3: S) =
struct
  include Printable.Prod3 (Base1) (Base2) (Base3)

  let bot () = (Base1.bot (), Base2.bot (), Base3.bot ())
  let top () = (Base1.top (), Base2.top (), Base3.top ())

  let leq (x1,x2,x3) (y1,y2,y3) = Base1.leq x1 y1 && Base2.leq x2 y2 && Base3.leq x3 y3

  let op_scheme op1 op2 op3 (x1,x2,x3) (y1,y2,y3): t = (op1 x1 y1, op2 x2 y2, op3 x3 y3)
  let join = op_scheme Base1.join Base2.join Base3.join
  let meet = op_scheme Base1.meet Base2.meet Base3.meet
end

module InfMap (Domain: Printable.S) (Range: S) =
struct 
  include Printable.InfMap (Domain) (Range)
    (* module M = HashTbl(Domain) *)

(*
  let bot (): t = (M.create 113, Range.bot ())
  let top (): t = (M.create 113, Range.top ())
*)
  let bot () : t = (M.empty, Range.bot ())
  let top () : t = (M.empty, Range.top ())
  let leq (xm,xd) ((ym,yd) as y) = 
    let fold_f key x_val accum = 
      if Range.leq x_val (find y key) then true else raise Result_known in
      Range.leq xd yd && try M.fold fold_f xm true with Result_known -> false

(*
  let op_scheme op (xm,xd) (ym,yd): t =
    let result = (M.create 113, op xd yd) in
    let resUpdate mapping key elem = 
      let elem' = find mapping key in
        replace result key (op elem' elem)
    in
      M.iter (resUpdate (xm,xd)) ym;
      M.iter (resUpdate (ym,yd)) xm;
      result
*)
  let op_scheme op ((xm,xd) as x) ((ym,yd) as y): t =
    let defval = op xd yd
    and mp = M.fold (fun k a -> M.add k (op a (find x k))) ym
             (M.fold (fun k a -> M.add k (op a (find y k))) xm M.empty)
  in (mp,defval)
  
  let join = op_scheme Range.join
  let meet = op_scheme Range.meet
end


module InfSet (Base: Printable.S) =
struct 
  include Printable.InfSet(Base)

  let bot () = empty
  let top () = All

  let leq = subset
  let join = union
  let meet = inter
end

module Sequence (Base: S) =
struct
  include Printable.Sequence (Base) 

  let bot () = [] 
  let top () = []

  let rec leq x y = 
    if List.length x = List.length y then begin
      match x,y with
        | [], [] -> true
        | (x::xs), (y::ys) -> Base.leq x y && leq xs ys
        | _ -> raise (Failure "This never happens you stupid compiler!")
    end else 
      false

  let rec op_scheme op x y: t =
    if List.length x = List.length y then begin
      match x, y with
        | [], [] -> []
        | (x::xs), (y::ys) -> op x y :: op_scheme op xs ys
        | _ -> raise (Failure "This never happens you stupid compiler!")
    end else
      raise (Failure "Different length")

  let join = op_scheme Base.join
  let meet = op_scheme Base.meet
end
