open Pretty

module type S =
sig
  include Datastruct.S
  val pretty: unit -> t -> doc
end

module Lift (Base: S)  =
struct
  include Datastruct.Lift(Base)
  let liftname = ref "order preserved"

  let bot_name = ref "bot"
  let top_name = ref "top"

  let pretty _ state = 
    match state with
      | Lifted n ->  Base.pretty () n
      | Bot -> text !bot_name
      | Top -> text !top_name

end

module Lift2 (Base1: S) (Base2: S) =
struct
  include Datastruct.Lift2 (Base1) (Base2)
  let liftname = ref "order preserved"

  let bot_name = ref "bot"
  let top_name = ref "top"

  let pretty _ state = 
    match state with
      | Lifted1 n ->  Base1.pretty () n
      | Lifted2 n ->  Base2.pretty () n
      | Bot -> text !bot_name
      | Top -> text !top_name

end

module Prod (Base1: S) (Base2: S) =
struct 
  open Pretty
  include Datastruct.Prod (Base1) (Base2)
  let pretty _ (x,y) = text "(" ++ Base1.pretty () x ++ text ", " ++ Base2.pretty () y ++ text ")"
end

module Prod3 (Base1: S) (Base2: S) (Base3: S) =
struct 
  open Pretty
  include Datastruct.Prod3 (Base1) (Base2) (Base3)
  let pretty _ (x,y,z) = text "(" ++ Base1.pretty () x ++ text ", " ++ Base2.pretty () y ++ text ", " ++ Base3.pretty () z ++ text ")"
end

module InfMap (Domain: S) (Range: S) =
struct
  include Datastruct.InfMap (Domain) (Range)

  let pretty () (mapping, defval) = 
    let f key st dok = 
      dok ++ (dprintf "%a -> \n  @[%a@]\n") Domain.pretty key Range.pretty st 
    in
    let content () = M.fold f mapping nil in
    let defline () = dprintf "OTHERS -> %a\n" Range.pretty defval in
      dprintf "@[Mapping {\n  @[%t%t@]}@]" content defline
end

module InfSet (Base: S)  = 
struct
  include Datastruct.InfSet (Base) 
  let all_name = ref "Complete set"

  let pretty () x = 
    match x with 
      | All -> text "Complete set"
      | Set x -> 
          let elts = S.elements x in
          let content = List.map (Base.pretty ()) elts in
          let rec separate x =
            match x with
              | [] -> []
              | [x] -> [x]
              | (x::xs) -> x ++ (text ", ") :: separate xs
          in 
          let separated = separate content in
          let content = List.fold_left (++) nil separated in
            (text "{") ++ content ++ (text "}")

end

module Sequence (Base: S): S with type t = Base.t list = 
struct 
  include Datastruct.Sequence (Base)

  let pretty () x =
    let rec isps x l = match l with
      [] | [_] -> l
    | (y :: ys) -> y :: x :: (isps x ys)
  in (text "[") ++ (List.fold_left Pretty.concat Pretty.nil (isps (text ", ") (List.map (Base.pretty ()) x))) ++ (text "]")
end
