(** A very fast demand-driven constraint solver *)

module Make (Var: Hashtbl.HashedType) (Dom: Lattice.S) :
sig
  module HT : Hash.S with type key = Var.t
  type 'a table    = 'a HT.t
  (** A hashtable from variables to ['a] *)

  (** A few type synonyms to make it more readable: *)
  type variable    = Var.t
  type domain      = Dom.t

  type assignment  = variable -> domain
  (** Assignments from variables to their values *)
  type assignment' = domain table 
  (** data structure representation of an assignment *)
  type rhs         = assignment -> domain 
  (** RHS of the constraint in functional form. The rhs is a function that given
    * the current values of the variables will evaluate the rhs expression *)
  type lhs         = variable 
  (** The lhs of each constraint is just a system variable *)
  type system      = lhs -> rhs list 
  (** A system is just a map from variables to the list of rh-sides constraining
    * each variable. *)

  type solution    = assignment'
  (** The solution to a system of constraints is an an assignment of variables
    * satisfying all constraints. We prefer a the data-structure representation
    * of the solution as a hashtable rather than an abstract mapping. *)

  val initstate: (variable -> domain) ref
  val solve: system -> variable list -> solution
  (** [solve system xs] solves a constraint [system] in a demand-driven fashion
    * starting from the variables [xs] *)
end 
